// service_worker.js

// ⚠️ MUST BE FIRST — loads syncClient before anything else
importScripts('sync-client.js');

// ── Receive session from verify page ──
chrome.runtime.onMessageExternal.addListener((message, sender, sendResponse) => {
  if (message.type === 'SESSION_RECEIVED' && message.session_id) {
    syncClient.saveSession(message.session_id).then(() => {
      console.log('[LinkedVue] Session saved from verify page');
      sendResponse({ ok: true });
    });
    return true; // keep channel open for async response
  }
});

//----------------------------------------------------------
// CREATE CONTEXT MENU ON ALL LINKEDIN LINKS
//----------------------------------------------------------
chrome.runtime.onInstalled.addListener(() => {
  chrome.contextMenus.create({
    id: "openCleanCRM",
    title: "Open with LinkedIn CRM (Clean URL)",
    contexts: ["link"]
  });

  chrome.alarms.create("dailyCleanup", { periodInMinutes: 1440 });
  chrome.alarms.create("linkedvueSyncAlarm", { periodInMinutes: 120 });
});

//----------------------------------------------------------
// CLEAN URL HANDLER
//----------------------------------------------------------
function cleanLinkedInUrl(raw) {
  try {
    if (raw.startsWith("/in/")) {
      return "https://www.linkedin.com" + raw.replace(/\/+$/, "");
    }
    const u = new URL(raw);
    if (!u.pathname.startsWith("/in/")) return null;
    const cleanPath = u.pathname.replace(/\/+$/, "");
    return `${u.origin}${cleanPath}`;
  } catch (e) {
    return null;
  }
}

//----------------------------------------------------------
// RIGHT-CLICK MENU ACTION
//----------------------------------------------------------
chrome.contextMenus.onClicked.addListener((info, tab) => {
  if (info.menuItemId !== "openCleanCRM") return;
  if (!info.linkUrl) return;

  if (!info.linkUrl.includes("linkedin.com/in/")) {
    chrome.notifications.create({
      type: "basic",
      iconUrl: "icons/icon128.png",
      title: "Not a LinkedIn Profile",
      message: "This option works only on LinkedIn profile URLs."
    });
    return;
  }

  const clean = info.linkUrl.split("?")[0].replace(/\/+$/, "");
  chrome.tabs.create({ url: clean, active: false });
});

//----------------------------------------------------------
// ALARMS HANDLER
//----------------------------------------------------------
chrome.alarms.onAlarm.addListener((alarm) => {

  if (alarm.name === "linkedvueSyncAlarm") {
    syncClient.scheduledSync();
    return;
  }

  if (alarm.name === "dailyCleanup") {
    const cutoff = new Date();
    cutoff.setDate(cutoff.getDate() - 365);

    chrome.storage.local.get("profiles", (res) => {
      const profiles = res.profiles || {};
      const updated = {};
      let removed = 0;

      for (const [url, p] of Object.entries(profiles)) {
        const captured = p.dateCaptured ? new Date(p.dateCaptured) : null;
        if (!captured || captured < cutoff) { removed++; continue; }
        updated[url] = p;
      }

      chrome.storage.local.set({ profiles: updated });
      console.log(`LinkedIn CRM auto-clean → removed ${removed} old records`);
    });
    return;
  }

  if (!alarm.name.startsWith("schedule_")) return;

  let decodedUrl;
  try {
    decodedUrl = atob(alarm.name.replace("schedule_", ""));
  } catch (e) {
    console.error("Invalid scheduled alarm name:", alarm.name);
    return;
  }

  if (!decodedUrl.startsWith("https://www.linkedin.com/in/")) {
    console.warn("Blocked scheduled open for invalid URL:", decodedUrl);
    return;
  }

  chrome.storage.local.get("profiles", (res) => {
    const profiles = res.profiles || {};
    const profile = profiles[decodedUrl];

    if (!profile || !profile.profileUrl) {
      console.warn("Scheduled profile not found:", decodedUrl);
      return;
    }

    chrome.tabs.create({ url: profile.profileUrl }, (tab) => {
      chrome.scripting.executeScript({
        target: { tabId: tab.id },
        files: ["chime.js"]
      });
      profiles[decodedUrl].status = "opened";
      chrome.storage.local.set({ profiles });
    });
  });

});