// onboarding.js

class LinkedVueOnboarding {
  constructor(flowConfig) {
    this.flows = flowConfig;
    this.currentFlow = null;
    this.currentStepIndex = 0;
    this.stateKey = "lv_onboarding_state";

    this.overlay = document.getElementById("lvOnboardingOverlay");
    this.card = document.getElementById("lvOnboardingCard");
    this.highlightBox = document.getElementById("lvHighlightBox");

    this.titleEl = document.getElementById("lvOnboardingTitle");
    this.textEl = document.getElementById("lvOnboardingText");
    this.stepNumEl = document.getElementById("lvStepNum");

    this.nextBtn = document.getElementById("lvOnboardingNext");
    this.prevBtn = document.getElementById("lvOnboardingPrev");
    this.skipBtn = document.getElementById("lvOnboardingSkip");
    this.closeBtn = document.getElementById("lvOnboardingClose");

    this._bindEvents();
  }

  async _getState() {
    return new Promise(resolve => {
      chrome.storage.local.get(this.stateKey, result => {
        resolve(result[this.stateKey] || null);
      });
    });
  }

  async _setState(state) {
    return new Promise(resolve => {
      chrome.storage.local.set({ [this.stateKey]: state }, resolve);
    });
  }

  async start(flowName) {
    const flow = this.flows[flowName];
    if (!flow) return;

    this.currentFlow = flowName;
    this.currentStepIndex = 0;

    await this._setState({
      activeFlow: flowName,
      stepIndex: 0,
      completed: false
    });

    this._showStep();
  }

  async resumeIfActive() {
    const state = await this._getState();
    if (!state || state.completed) return;

    const flow = this.flows[state.activeFlow];
    if (!flow) return;

    this.currentFlow = state.activeFlow;
    this.currentStepIndex = state.stepIndex;

    this._showStep();
  }

  async next() {
    const flow = this.flows[this.currentFlow];
    if (!flow) return;

    if (this.currentStepIndex < flow.length - 1) {
      this.currentStepIndex++;
      await this._setState({
        activeFlow: this.currentFlow,
        stepIndex: this.currentStepIndex,
        completed: false
      });
      this._showStep();
    } else {
      this.complete();
    }
  }

  async prev() {
    if (this.currentStepIndex > 0) {
      this.currentStepIndex--;
      await this._setState({
        activeFlow: this.currentFlow,
        stepIndex: this.currentStepIndex,
        completed: false
      });
      this._showStep();
    }
  }

  async skip() {
    await this.complete();
  }

  async complete() {
    await this._setState({
      activeFlow: null,
      stepIndex: 0,
      completed: true
    });

    this._hideOverlay();
  }

  async reset() {
    await chrome.storage.local.remove(this.stateKey);
  }

  async _showStep() {
    const step = this.flows[this.currentFlow][this.currentStepIndex];
    if (!step) return;

    this.overlay.classList.remove("hidden");
    this.overlay.classList.remove("center-mode");

    this.titleEl.innerText = step.title || "";
    this.textEl.innerText = step.text || "";
    this.stepNumEl.innerText = this.currentStepIndex + 1;

    this.prevBtn.classList.toggle("hidden", this.currentStepIndex === 0);

    if (step.mode === "center") {
      this.overlay.classList.add("center-mode");
      this.highlightBox.style.display = "none";
    } else if (step.selector) {
      await this._highlightElement(step.selector);
    }

    if (step.onNext) {
      this.nextBtn.onclick = async () => {
        await step.onNext();
        this.next();
      };
    } else {
      this.nextBtn.onclick = () => this.next();
    }
  }

  async _highlightElement(selector) {
    const el = await this._waitForElement(selector);
    if (!el) return;

    const rect = el.getBoundingClientRect();

    this.highlightBox.style.display = "block";
    this.highlightBox.style.top = rect.top - 6 + "px";
    this.highlightBox.style.left = rect.left - 6 + "px";
    this.highlightBox.style.width = rect.width + 12 + "px";
    this.highlightBox.style.height = rect.height + 12 + "px";

    this._positionCard(rect);

    el.scrollIntoView({ behavior: "smooth", block: "center" });
  }

  _positionCard(rect) {
    const cardWidth = this.card.offsetWidth;
    const cardHeight = this.card.offsetHeight;

    let top = rect.bottom + 14;
    let left = rect.left;

    if (left + cardWidth > window.innerWidth) {
      left = window.innerWidth - cardWidth - 16;
    }

    if (top + cardHeight > window.innerHeight) {
      top = rect.top - cardHeight - 14;
    }

    this.card.style.top = top + "px";
    this.card.style.left = left + "px";
  }

  _hideOverlay() {
    this.overlay.classList.add("hidden");
  }

  _bindEvents() {
    this.skipBtn.onclick = () => this.skip();
    this.closeBtn.onclick = () => this.skip();
    this.prevBtn.onclick = () => this.prev();
  }

  _waitForElement(selector, timeout = 5000) {
    return new Promise(resolve => {
      const interval = 100;
      let elapsed = 0;

      const timer = setInterval(() => {
        const el = document.querySelector(selector);
        if (el) {
          clearInterval(timer);
          resolve(el);
        }
        elapsed += interval;
        if (elapsed >= timeout) {
          clearInterval(timer);
          resolve(null);
        }
      }, interval);
    });
  }
}