// sync-client.js — LinkedVue Sync Client

const SYNC_CONFIG = {
  apiBase: 'https://api.linkedvue.com',
  stalePullMs: 4 * 60 * 60 * 1000,
  sessionKey: 'linkedvue_session_id',
  lastSyncKey: 'linkedvue_last_sync',
  lastPullKey: 'linkedvue_last_pull',
  syncStateKey: 'linkedvue_sync_state', // tracks initial sync progress
};

// ─── SESSION ──────────────────────────────────────────────────────────────────

async function getSession() {
  const data = await chrome.storage.local.get(SYNC_CONFIG.sessionKey);
  return data[SYNC_CONFIG.sessionKey] || null;
}

async function isAuthenticated() {
  return !!(await getSession());
}

async function saveSession(sessionId) {
  await chrome.storage.local.set({
    [SYNC_CONFIG.sessionKey]: sessionId,
    [SYNC_CONFIG.lastPullKey]: 0,
  });
}

async function clearSession() {
  await chrome.storage.local.remove([
    SYNC_CONFIG.sessionKey,
    SYNC_CONFIG.lastSyncKey,
    SYNC_CONFIG.lastPullKey,
    SYNC_CONFIG.syncStateKey,
  ]);
}

// ─── API CALL ─────────────────────────────────────────────────────────────────

async function apiCall(method, path, body = null) {
  const session = await getSession();
  if (!session) throw new Error('Not authenticated');

  const options = {
    method,
    headers: {
      'Content-Type': 'application/json',
      'Authorization': `Bearer ${session}`,
    },
  };
  if (body) options.body = JSON.stringify(body);

  const response = await fetch(`${SYNC_CONFIG.apiBase}${path}`, options);
  const data = await response.json();
  if (!response.ok) throw new Error(data.error || 'API error');
  return data;
}

// ─── BUILD PUSH PAYLOAD ───────────────────────────────────────────────────────

async function buildPushPayload(triggeredBy = 'scheduled') {
  const storage = await chrome.storage.local.get(['profiles', 'templates', 'icpSettings']);

  const rawProfiles  = storage.profiles  || {};
  const rawTemplates = storage.templates || {};
  const icpSettings  = storage.icpSettings || null;

  const profiles = Object.entries(rawProfiles).map(([url, p]) => ({
    profile_url: url,
    first_name: p.firstName || null,
    last_name:  p.lastName  || null,
    email:      p.email     || null,
    notes:      p.notes     || null,
    status:     p.status    || null,
    icp_score:  p.icpScore  || null,
    raw_profile_text:    p.rawProfileText    || null,
    ai_message:          p.aiMessage         || null,
    template_id:         p.templateUsed      || null,
    sequence_delay_days: p.sequenceDelayDays || null,
    next_due_date: p.nextDueDate ? new Date(p.nextDueDate).getTime() : null,
    created_at:    p.dateCaptured ? new Date(p.dateCaptured).getTime() : Date.now(),
    updated_at:    p.lastUpdated  ? new Date(p.lastUpdated).getTime()  : Date.now(),
    messages: {
      msg1: { text: p.messages?.msg1 || null, sentAt: p.sentDates?.msg1 ? new Date(p.sentDates.msg1).getTime() : null },
      msg2: { text: p.messages?.msg2 || null, sentAt: p.sentDates?.msg2 ? new Date(p.sentDates.msg2).getTime() : null },
      msg3: { text: p.messages?.msg3 || null, sentAt: p.sentDates?.msg3 ? new Date(p.sentDates.msg3).getTime() : null },
    }
  }));

  const templates = Object.entries(rawTemplates).map(([name, t]) => ({
    id:         t.id   || name,
    name:       t.name || name,
    delay_days: t.delayDays || 3,
    created_at: t.createdAt ? new Date(t.createdAt).getTime() : Date.now(),
    updated_at: t.updatedAt ? new Date(t.updatedAt).getTime() : Date.now(),
    messages: {
      msg1: t.messages?.msg1 || null,
      msg2: t.messages?.msg2 || null,
      msg3: t.messages?.msg3 || null,
    }
  }));

  const icp = icpSettings ? {
    industries: icpSettings.industries || [],
    regions:    icpSettings.regions    || [],
    personas:   icpSettings.personas   || [],
    goal:       icpSettings.goal       || null,
    offer:      icpSettings.offer ? JSON.stringify(icpSettings.offer) : null,
  } : null;

  return { profiles, templates, icp_settings: icp, triggered_by: triggeredBy };
}

// ─── PUSH ─────────────────────────────────────────────────────────────────────

async function push(triggeredBy = 'scheduled') {
  try {
    const auth = await isAuthenticated();
    if (!auth) {
      console.log('[LinkedVue Sync] Not authenticated — skipping push');
      return { skipped: true };
    }

    const payload = await buildPushPayload(triggeredBy);
    if (!payload.profiles.length && !payload.templates.length && !payload.icp_settings) {
      console.log('[LinkedVue Sync] Nothing to push');
      return { skipped: true };
    }

    const result = await apiCall('POST', '/sync/push', payload);

    // Handle limit reached
    if (result.limit_reached) {
      console.warn('[LinkedVue Sync] Profile limit reached:', result.current_count);
      await chrome.storage.local.set({ linkedvue_limit_reached: true });
    }

    await chrome.storage.local.set({ [SYNC_CONFIG.lastSyncKey]: Date.now() });
    console.log('[LinkedVue Sync] Push complete:', result.synced);
    return result;

  } catch (err) {
    console.error('[LinkedVue Sync] Push failed:', err.message);
    return { error: err.message };
  }
}

// ─── PULL (paginated) ─────────────────────────────────────────────────────────

async function pull(onProgress = null) {
  try {
    const auth = await isAuthenticated();
    if (!auth) return { skipped: true };

    const lastPullData = await chrome.storage.local.get(SYNC_CONFIG.lastPullKey);
    const since = lastPullData[SYNC_CONFIG.lastPullKey] || 0;
    const isFirstPull = since === 0;

    // Check server state first to know total count
    let serverTotal = 0;
    if (isFirstPull) {
      try {
        const state = await apiCall('GET', '/sync/state');
        serverTotal = state.profiles || 0;
        if (serverTotal > 0) {
          // Save sync state so dashboard can show progress banner
          await chrome.storage.local.set({
            [SYNC_CONFIG.syncStateKey]: {
              status: 'syncing',
              total: serverTotal,
              loaded: 0,
              startedAt: Date.now()
            }
          });
          if (onProgress) onProgress({ status: 'started', total: serverTotal, loaded: 0 });
        }
      } catch(e) {}
    }

    // Paginated pull
    let page = 0;
    let hasMore = true;
    let allProfiles = [];
    let templates = [];
    let serverTime = Date.now();
    let totalLoaded = 0;

    while (hasMore) {
      let result;
      try {
        result = await apiCall('GET', `/sync/pull?since=${since}&page=${page}`);
      } catch(e) {
        // Save partial progress so user can retry
        await chrome.storage.local.set({
          [SYNC_CONFIG.syncStateKey]: {
            status: 'error',
            total: serverTotal,
            loaded: totalLoaded,
            error: e.message,
            since,
            page
          }
        });
        if (onProgress) onProgress({ status: 'error', error: e.message, loaded: totalLoaded, total: serverTotal });
        console.error('[LinkedVue Sync] Pull page failed:', e.message);
        return { error: e.message, partial: true, loaded: totalLoaded };
      }

      allProfiles = allProfiles.concat(result.profiles || []);
      if (page === 0) templates = result.templates || [];
      serverTime = result.server_time || serverTime;
      hasMore    = result.has_more || false;
      totalLoaded = result.loaded || allProfiles.length;
      page++;

      // Update progress
      if (onProgress) onProgress({ status: 'syncing', total: result.total || serverTotal, loaded: totalLoaded });
      await chrome.storage.local.set({
        [SYNC_CONFIG.syncStateKey]: {
          status: 'syncing',
          total: result.total || serverTotal,
          loaded: totalLoaded
        }
      });

      if (page > 20) break; // safety cap
    }

    // Merge profiles into local storage
    if (allProfiles.length > 0) {
      const existing = await chrome.storage.local.get('profiles');
      const currentProfiles = existing.profiles || {};

      for (const profile of allProfiles) {
        if (profile.deleted_at) {
          delete currentProfiles[profile.profile_url];
          continue;
        }
        const existingRecord = currentProfiles[profile.profile_url] || {};
        currentProfiles[profile.profile_url] = {
          ...existingRecord,
          profileUrl:          profile.profile_url,
          // Never overwrite manually entered data with null from DB
          firstName:           existingRecord.firstName  || profile.first_name  || null,
          lastName:            existingRecord.lastName   || profile.last_name   || null,
          email:               existingRecord.email      || profile.email       || null,
          notes:               existingRecord.notes      || profile.notes       || null,
          status:              profile.status            || existingRecord.status,
          icpScore:            profile.icp_score         || existingRecord.icpScore,
          rawProfileText:      existingRecord.rawProfileText || profile.raw_profile_text,
          aiMessage:           existingRecord.aiMessage  || profile.ai_message,
          templateUsed:        profile.template_id       || existingRecord.templateUsed,
          sequenceDelayDays:   profile.sequence_delay_days || existingRecord.sequenceDelayDays,
          nextDueDate:         profile.next_due_date ? new Date(profile.next_due_date).toISOString() : existingRecord.nextDueDate || null,
          dateCaptured:        existingRecord.dateCaptured || new Date(profile.created_at).toISOString(),
          lastUpdated:         new Date(profile.updated_at).toISOString(),
          messages: {
            msg1: profile.messages.find(m => m.slot === 1)?.content || existingRecord.messages?.msg1 || null,
            msg2: profile.messages.find(m => m.slot === 2)?.content || existingRecord.messages?.msg2 || null,
            msg3: profile.messages.find(m => m.slot === 3)?.content || existingRecord.messages?.msg3 || null,
          },
          sentDates: {
            msg1: existingRecord.sentDates?.msg1 || (profile.messages.find(m => m.slot === 1)?.sent_at ? new Date(profile.messages.find(m => m.slot === 1).sent_at).toISOString() : null),
            msg2: existingRecord.sentDates?.msg2 || (profile.messages.find(m => m.slot === 2)?.sent_at ? new Date(profile.messages.find(m => m.slot === 2).sent_at).toISOString() : null),
            msg3: existingRecord.sentDates?.msg3 || (profile.messages.find(m => m.slot === 3)?.sent_at ? new Date(profile.messages.find(m => m.slot === 3).sent_at).toISOString() : null),
          }
        };
      }

      await chrome.storage.local.set({ profiles: currentProfiles });
      console.log('[LinkedVue Sync] Pull merged:', allProfiles.length, 'profiles across', page, 'pages');
    }

    await chrome.storage.local.set({ [SYNC_CONFIG.lastPullKey]: serverTime });

    // Mark sync complete
    await chrome.storage.local.set({
      [SYNC_CONFIG.syncStateKey]: {
        status: 'complete',
        total: serverTotal || allProfiles.length,
        loaded: totalLoaded || allProfiles.length,
        completedAt: Date.now()
      }
    });
    if (onProgress) onProgress({ status: 'complete', total: serverTotal, loaded: totalLoaded || allProfiles.length });

    return { profiles: allProfiles, templates };

  } catch (err) {
    console.error('[LinkedVue Sync] Pull failed:', err.message);
    await chrome.storage.local.set({
      [SYNC_CONFIG.syncStateKey]: { status: 'error', error: err.message }
    });
    if (onProgress) onProgress({ status: 'error', error: err.message });
    return { error: err.message };
  }
}

// ─── PULL IF STALE ────────────────────────────────────────────────────────────

async function pullIfStale(onProgress = null) {
  const auth = await isAuthenticated();
  if (!auth) return;

  const data = await chrome.storage.local.get(SYNC_CONFIG.lastPullKey);
  const lastPull = data[SYNC_CONFIG.lastPullKey] || 0;
  const isStale  = Date.now() - lastPull > SYNC_CONFIG.stalePullMs;

  if (isStale) {
    console.log('[LinkedVue Sync] Data is stale — pulling from server');
    await pull(onProgress);
  }
}

// ─── EVENT PUSH (debounced) ───────────────────────────────────────────────────

let pushDebounceTimer = null;
function pushEvent(triggeredBy = 'event') {
  clearTimeout(pushDebounceTimer);
  pushDebounceTimer = setTimeout(() => push(triggeredBy), 5000);
}

// ─── SCHEDULED SYNC ──────────────────────────────────────────────────────────

async function scheduledSync() {
  const auth = await isAuthenticated();
  if (!auth) return;
  console.log('[LinkedVue Sync] Running scheduled sync');
  await push('scheduled');
  pull();
}

// Export
const syncClient = {
  push, pull, pullIfStale, pushEvent, scheduledSync,
  isAuthenticated, saveSession, clearSession, getSession,
};